/*
 * FPSE linux CD routines.
 *
 * By: linuzappz <linuzappz@hotmail.com>
 *
 * STATUS: Works fine, no buffering is used here, as FPSE has its own buffering
 *	   routines.
 *
 */

#include <stdio.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include <sys/stat.h>
#include <unistd.h>

#include "type.h"
#include "cd.h"
#include "linuxdef.h"

#define VERSION		0x0100  // Version must be in BCD format

static FPSELinuxAbout info = {
    FPSE_CDROM,
    VERSION & 0xFF,
    VERSION >> 8,
    0,
    "LinuzAppz",
    "Linux-CDROM",
    "This Plugin is a CD driver. It is based on my last SOPE CD driver.\x00d\x00a"
    "If you have some question or you've found a bug send an email to:\x00d\x00a"
    "<linuzappz@hotmail.com>"
};


#define CDROM_DEV	"/dev/cdrom"
#define DATA_SIZE	(CD_FRAMESIZE_RAW-12)

static UINT8 cdbuffer[DATA_SIZE];
static int cdHandle = -1;

int CD_Open(UINT32 *par)
{
	if (cdHandle > 0) return FPSE_OK; /* it's already open */
	cdHandle = open(CDROM_DEV, O_RDONLY | O_NONBLOCK);
	if (cdHandle == -1)
	    {
	    printf ("FPSE: could not open %s\n",CDROM_DEV);
	    return FPSE_ERR;
	    }
	if (ioctl(cdHandle,CDROMSTART,NULL))
	    {
	    printf ("FPSE: Error starting cdrom at %s\n",CDROM_DEV);
	    return FPSE_ERR;
	    }

	CD_Stop();
	return FPSE_OK;
}

void CD_Close(void)
{
	if (cdHandle < 1) return;
	CD_Stop();
	close(cdHandle);
	cdHandle = -1;
}

int CD_GetTN(UINT8 *result)
{
	struct cdrom_tochdr toc;

	if (cdHandle < 1) return FPSE_ERR;

	if (ioctl(cdHandle,CDROMREADTOCHDR,&toc) == -1) return FPSE_ERR;

	result[1] = toc.cdth_trk0;
	result[2] = toc.cdth_trk1;
  
	return FPSE_OK;
}

int CD_GetTD(UINT8 *result,int track)
{
	struct cdrom_tocentry entry;

	if (cdHandle < 1) return FPSE_ERR;

	entry.cdte_track = ++track;
	entry.cdte_format = CDROM_MSF;

	if (ioctl(cdHandle, CDROMREADTOCENTRY, &entry) == -1) return FPSE_ERR;

        result[1] = entry.cdte_addr.msf.minute; /* minute */
	result[2] = entry.cdte_addr.msf.second; /* second */
        result[3] = entry.cdte_addr.msf.frame; /* frame */

	return FPSE_OK;
}

int CD_Play(UINT8 *param)
{
	struct cdrom_msf addr;
	UINT8 ptmp[4];

        if (cdHandle < 1) return FPSE_ERR;

	if (CD_GetTN(ptmp) == FPSE_ERR) return FPSE_ERR;
	if (CD_GetTD(ptmp, ptmp[2]-1) == FPSE_ERR) return FPSE_ERR; 
	addr.cdmsf_min0 = param[0];
	addr.cdmsf_sec0 = param[1];
	addr.cdmsf_frame0 = param[2];
        addr.cdmsf_min1 = ptmp[1];
	addr.cdmsf_sec1 = ptmp[2];
        addr.cdmsf_frame1 = ptmp[3];

	if (ioctl(cdHandle, CDROMPLAYMSF, &addr) == -1)
	    {
    	    printf("CD_Play(): Error Playing CD-DA: min=%d, sec=%d, frame=%d\n",
               addr.cdmsf_min0, addr.cdmsf_sec0, addr.cdmsf_frame0);
	    return FPSE_ERR;
	    }

	return FPSE_OK;
}

int CD_Stop(void)
{
        struct cdrom_subchnl sc;

	if (cdHandle < 1) return FPSE_ERR;

	sc.cdsc_format = CDROM_MSF;
        if (ioctl(cdHandle, CDROMSUBCHNL, &sc) == -1) return FPSE_ERR;

	switch (sc.cdsc_audiostatus) 
	    {
    	    case CDROM_AUDIO_PAUSED: 
    	    case CDROM_AUDIO_PLAY: ioctl(cdHandle, CDROMSTOP); break;
	    }

	return FPSE_OK;
}

UINT8 *CD_Read(UINT8 *param)
{
	union {
	    struct cdrom_msf msf;
	    UINT8 buf[CD_FRAMESIZE_RAW];
	} cr;

        if (cdHandle < 1) return NULL;

	cr.msf.cdmsf_min0 = param[0];
	cr.msf.cdmsf_sec0 = param[1];
	cr.msf.cdmsf_frame0 = param[2];

	if (ioctl(cdHandle, CDROMREADRAW, &cr)) {
	    printf ("FPSE: Error reading cdrom data\n");
	    return NULL;
	}

	memcpy(cdbuffer, cr.buf+12, CD_FRAMESIZE_RAW-12); /* skip sync data */

	return cdbuffer;
}

void CD_GetSeek(UINT8 *par)
{
	/* what to do here? */
}


/* This should wait until the sector is read */
/* As its already readed do nothing */
int CD_Wait()
{
	return FPSE_OK;
}

/* There is nothing to configure */
int CD_Configure(UINT32 *par)
{
	FPSELinux *conf = (FPSELinux *) par;

	conf->ConfCmd(0,NULL); /* start/reset */
	conf->ConfCmd(1,"Nothing to Configure"); /* print */
	conf->ConfCmd(0xf,NULL); /* wait and close */

	return FPSE_OK;
}


void CD_About(UINT32 *par)
{
	struct stat buf;

	if (stat(CDROM_DEV,&buf)==-1) info.TestResult = -1;
	memcpy (par, &info, sizeof(FPSELinuxAbout));
}